/*/////////////////////////////////////////////////////////////////////////////
//! \file fcBaseTypesCAN.h:
//! \brief Header file for the data types and enumerations of the fcBase CAN API.
//!
//! Use this header file for developing user defined FlexCard applications. This
//! header file contains all structures, data types and enumerations which are
//! used in the fcBase CAN API. For a description of the API function, please
//! refer to the header file "fcBaseCAN.h".
//!
//! See also printed documentation for more information.
//!
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-02-10 13:25:10 +0100 (Di, 10 Feb 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 14979 $
/////////////////////////////////////////////////////////////////////////////*/

#ifndef _FCBASE_TYPES_CAN_H_
#define _FCBASE_TYPES_CAN_H_

//-----------------------------------------------------------------------------
//           Initialization 
//-----------------------------------------------------------------------------

//! This enumeration defines the different modes available, used to monitor a
//! CAN cluster.
//! \see #fcbCANMonitoringStart
typedef enum fcCANMonitoringMode
{
	fcCANMonitoringNormal = 0,						//!< The FlexCard will switch from configuration to
	fcCANMonitoringActive = fcCANMonitoringNormal,	//!< normal active state as soon as possible. In normal
													//!< active state CAN frames can be received and transmitted.
	fcCANMonitoringSilent = 1,						//!< The FlexCard will switch from configuration to
	fcCANMonitoringPassive = fcCANMonitoringSilent,	//!< normal passive state as soon as possible. In normal
													//!< passive state CAN frames can be received only.
} fcCANMonitoringMode;

//-----------------------------------------------------------------------------
//          Configuration
//-----------------------------------------------------------------------------

//! For the transmission and reception of CAN frames the communication controller
//! provides different types of message buffers. Each message buffer can be
//! assigned with one of the following specific types.
//! \see #fcCANBufCfg
typedef enum fcCANBufCfgType
{
	fcCANBufCfgTypeNone = 0,	//!< The message buffer is not used. It can be used to reset
								//!< a message buffer.
	fcCANBufCfgTypeCommon,		//!< The message buffer is reserved for internal use only.
								//!< (No support.)
	fcCANBufCfgTypeRxAll,		//!< The message buffer is used for receiving all incoming
								//!< CAN data and remote frames.
	fcCANBufCfgTypeRx,			//!< The message buffer is used as a receive buffer for either
								//!< a specific message or a set of messages.
	fcCANBufCfgTypeTx,			//!< The message buffer is used as a transmit buffer for a
								//!< specific CAN message ID.
	fcCANBufCfgTypeRemoteRx,	//!< The message buffer is used as a remote receive buffer.
								//!< It is used for sending a remote request and receiving
								//!< the according replying message.
	fcCANBufCfgTypeRemoteTx,	//!< The message buffer is used as a remote transmission buffer.
								//!< It can be transmitted automatically when a remote request is received.
} fcCANBufCfgType;

//! This enumeration defines the acceptance conditions of an fcCANBufCfgRxAll 
//! buffer. To accept all incoming messages you need to binary OR all of them.
//! \see #fcCANBufCfgRxAll
typedef enum fcCANBufCfgRxAllCondition
{
	fcCANRxAllNone			= 0x0,	//!< Accept no frames.
	fcCANRxAllIDStandard	= 0x1,	//!< Accept CAN frames with standard identifiers.
	fcCANRxAllIDExtended	= 0x2,	//!< Accept CAN frames with extended identifiers.
	fcCANRxAllFrameData		= 0x4,	//!< Accept all CAN data frames.
	fcCANRxAllFrameRemote	= 0x8,	//!< Accept all CAN remote frames.
} fcCANBufCfgRxAllCondition;

//! This structure specifies the configuration of a CAN common message buffer.
//! This buffer type is used for internal purposes at the moment. No support.
typedef struct fcCANBufCfgCommon
{
	fcDword ID;
	fcDword MaskID;
	fcByte  Data[8];
	fcDword DLC					:4;
	fcDword enableIDExtended	:1;
	fcDword Direction			:1;
	fcDword MaskDirection		:1;
	fcDword enableMask			:1;
	fcDword enableRemote		:1;
	fcDword enableTxAcknowledge	:1;
	fcDword enableTxRequest		:1;
	fcDword newData				:1;
	fcDword Reserved[2];
} fcCANBufCfgCommon;

//! This structure specifies a special CAN receive message buffer. This buffer
//! type is used to receive all frames of the specified conditions.
//! \see #fcCANBufCfg, #fcCANBufCfgRxAllCondition
typedef struct fcCANBufCfgRxAll
{
	fcDword Condition;		//!< The acceptance condition for this buffer, which can be
							//!< OR-ed. At least one id condition and one frame condition
							//!< must be used to receive frames.
	fcDword Reserved[2];	//!< Reserved for future use.
} fcCANBufCfgRxAll;

//! This structure specifies the configuration of a CAN receive message buffer.
//! This buffer type is used to receive a CAN message with a specific CAN ID
//! only or a range of CAN IDs.
//! \see #fcCANBufCfg
typedef struct fcCANBufCfgRx
{
	fcDword ID;						//!< Defines the CAN identifier to be received in this
									//!< message buffer. Valid values for a standard CAN Id
									//!< range from 0x0 to 0x7FF. Valid values for an extended
									//!< CAN Id range from 0x0 to 0x1FFFFFFF.
	fcDword MaskID;					//!< The bit mask. The corresponding identifier bits are
									//!< used for acceptance filtering.
	fcDword enableIDExtended	:1;	//!< If set to 1 the message buffer is defined for extended
									//!< CAN identifiers only. If set to 0 the message buffer is
									//!< defined for standard CAN identifiers. It is not possible
									//!< to receive both versions in one message buffer.
	fcDword enableMask			:1;	//!< Set this flag to 1 to enable the acceptance mask.
	fcDword Reserved[2];			//!< Reserved for future use.
} fcCANBufCfgRx;

//! This structure specifies the configuration of a CAN transmit message buffer.
//! This buffer type is used to transmit a CAN message with a specific CAN ID only.
//! \see #fcCANBufCfg, #fcCANPacket
typedef struct fcCANBufCfgTx
{	
	fcDword ID;						//!< Defines the CAN identifier to be received in this
									//!< message buffer. Valid values for a standard CAN Id
									//!< range from 0x0 to 0x7FF. Valid values for an extended
									//!< CAN Id range from 0x0 to 0x1FFFFFFF.
	fcByte	Data[8];				//!< Defines the data for transmission. All of the 8 data
									//!< bytes can be set. The corresponding DLC parameter is
									//!< used to define the number of bytes to transmit.
	fcDword DLC					:4;	//!< Defines the data length (in bytes) to be transmitted.
	fcDword enableIDExtended	:1;	//!< If set to 1 the message buffer is defined for extended
									//!< CAN identifiers only. If set to 0 the message buffer is
									//!< defined for standard CAN identifiers. It is not possible
									//!< to receive both versions in one message buffer.
	fcDword enableTxAcknowledge	:1;	//!< Set this flag to 1 to enable the transmit acknowledge.
									//!< The FlexCard generates a CAN packet with a direction
									//!< flag = 1 (Tx), if the data are transmitted successfully.
	fcDword enableTxRequest		:1;	//!< Set this flag to 1 to indicate that the message is
									//!< requested to be sent as soon as the communication
									//!< controller is in state normal active.
	fcDword newData				:1;	//!< Set this flag to 1 to update the data of the message buffer.
									//!< Set to 0 if no new data shall be updated.
	fcDword Reserved[2];			//!< Reserved for future use.
} fcCANBufCfgTx;

//! This structure specifies a CAN remote receive message buffer. This buffer
//! type is used to send a CAN remote message to request a CAN message with
//! the same CAN identifier. This will be received into the message buffer.
//! \see #fcCANBufCfg, #fcCANPacket
typedef struct fcCANBufCfgRemoteRx 
{
	fcDword ID;						//!< Defines the CAN identifier to be received in this
									//!< message buffer. Valid values for a standard CAN Id
									//!< range from 0x0 to 0x7FF. Valid values for an extended
									//!< CAN Id range from 0x0 to 0x1FFFFFFF.
	fcDword DLC					:4;	//!< Defines the data length (in bytes) to be transmitted.
	fcDword enableIDExtended	:1;	//!< If set to 1 the message buffer is defined for extended
									//!< CAN identifiers only. If set to 0 the message buffer is
									//!< defined for standard CAN identifiers. It is not possible
									//!< to receive both versions in one message buffer.
	fcDword enableTxAcknowledge	:1;	//!< Set this flag to 1 to enable the transmit acknowledge.
									//!< The FlexCard generates a CAN packet (RemoteTx) if the
									//!< data are transmitted successfully.
	fcDword enableTxRequest		:1;	//!< Set this flag to 1 to indicate that the message is
									//!< requested to be sent as soon as the communication
									//!< controller is in state normal active.
	fcDword Reserved[2];			//!< Reserved for future use.
} fcCANBufCfgRemoteRx;

//! This structure specifies a CAN remote transmit message buffer. This buffer
//! type is used to transmit a CAN message when this Id is requested by a
//! corresponding CAN remote frame.
//! \see #fcCANBufCfg, #fcCANPacket
typedef struct fcCANBufCfgRemoteTx 
{
	fcDword ID;						//!< Defines the CAN identifier to be received in this
									//!< message buffer. Valid values for a standard CAN Id
									//!< range from 0x0 to 0x7FF. Valid values for an extended
									//!< CAN Id range from 0x0 to 0x1FFFFFFF.
	fcByte  Data[8];				//!< Defines the data for transmission. All of the 8 data
									//!< bytes can be set. The corresponding DLC parameter is
									//!< used to define the number of bytes to transmit.
	fcDword DLC					:4;	//!< Defines the data length (in bytes) to be transmitted.
	fcDword enableIDExtended	:1;	//!< If set to 1 the message buffer is defined for extended
									//!< CAN identifiers only. If set to 0 the message buffer is
									//!< defined for standard CAN identifiers. It is not possible
									//!< to receive both versions in one message buffer.
	fcDword enableTxAcknowledge	:1;	//!< Set this flag to 1 to enable the transmit acknowledge.
									//!< The FlexCard generates a CAN packet (RemoteTx) if the
									//!< data are transmitted successfully and the parameter
									//!< enableAutoResponse is set to 1 too.
	fcDword enableTxRequest		:1;	//!< Set this flag to 1 to indicate that the message is
									//!< requested to be sent as soon as the communication
									//!< controller is in state normal active.
	fcDword enableAutoResponse	:1;	//!< Set this flag to 1 to enable the buffer to transmit a frame
									//!< as soon as a corresponding CAN remote frame is received.
	fcDword newData				:1;	//!< Set this flag to 1 to update the data of the message buffer.
									//!< Set to 0 if no new data shall be updated.
	fcDword Reserved[2];			//!< Reserved for future use.
} fcCANBufCfgRemoteTx;

//! This structure describes the configuration of a CAN message buffer.
//! \see #fcCANBufCfgType, #fcCANBufCfgRxAll, #fcCANBufCfgRx,
//!		 #fcCANBufCfgTx, #fcCANBufCfgRemoteRx, #fcCANBufCfgRemoteTx,
//!		 #fcbCANSetMessageBuffer, #fcbCANGetMessageBuffer
typedef struct fcCANBufCfg
{
	fcCANBufCfgType Type;				//!< Defines the CAN message buffer type.
	union 
	{
		fcCANBufCfgCommon Common;		//!< Used for internal purposes. (No support).
		fcCANBufCfgRxAll RxAll;			//!< Receive all buffer configuration.
		fcCANBufCfgRx Rx;				//!< Receive buffer configuration.
		fcCANBufCfgTx Tx;				//!< Transmit buffer configuration.
		fcCANBufCfgRemoteRx RemoteRx;	//!< Remote receive buffer configuration.
		fcCANBufCfgRemoteTx RemoteTx;	//!< Remote transmit buffer configuration.
	};
} fcCANBufCfg;

//! This structure describes the configuration of a CAN communication controller.
//! \see #fcCANSetCcConfiguration
//! \remarks The baud rate and the sample point calculation by the CAN communication
//!			 controller depends on BaudRatePrescaler, SynchronizationJumpWidth,
//!			 TimeSegment1 and TimeSegment2.
//!			 Baud rate [baud] = 16 * 10^6 [Hz] / ((3 + TSEG1 + TSEG2) * (1 + BRP))
//!			 Sample point [%] = 100 * (2 + TSEG1) / (3 + TSEG1 + TSEG2)
typedef struct fcCANCcConfig
{
	fcWord BaudRatePrescaler;					//!< Defines the baud rate prescaler (BRP). Valid values
												//!< are from 0 to 1023.
	fcWord SynchronizationJumpWidth;			//!< Defines the synchronization jump width (SJW). Valid
												//!< values are from 0 to 3 and must not be larger than
												//!< TSEG1 and TSEG2.
	fcWord TimeSegment1;						//!< Defines the time segment 1 (TSEG1). Valid values
												//!< are from 0 to 15.
	fcWord TimeSegment2;						//!< Defines the time segment 2 (TSEG2). Valid values
												//!< are from 0 to 7.
	fcDword enableAutomaticRetransmission	:1;	//!< Set this flag to 1 to enable automatic retransmission.
												//!< If the CAN communication controller has lost the arbitration
												//!< or if an error occurred during the transmission, the message
												//!< will be retransmitted as soon as the CAN bus is free again.
	fcDword Reserved[6];						//!< Reserved for future use.
} fcCANCcConfig;

#endif  /* #define _FCBASE_TYPES_CAN_H_ */
