///////////////////////////////////////////////////////////////////////////////
//! \file fcBaseTypesFlexRay.h:
//! \brief Header file for the data types and enumerations of the fcBase
//!        FlexRay API.
//!
//! Use this header file for developing user defined FlexCard applications. This
//! header file contains all structures, data types and enumerations which are
//! used in the fcBase FlexRay API. For a description of the API function, please
//! refer to the header file "fcBaseFlexRay.h".
//!
//! See also printed documentation for more information. 
//!  
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-01-29 10:06:09 +0100 (Do, 29 Jan 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 14920 $
///////////////////////////////////////////////////////////////////////////////

#ifndef _FCBASE_TYPES_FLEXRAY_H_
#define _FCBASE_TYPES_FLEXRAY_H_

//! This enumeration contains all supported FlexRay baud rates.
//! \see #fcFRCcConfig
typedef enum fcFRBaudRate
{
	//! No FlexRay baud rate
	fcFRBaudRateNone = 0,
	//! FlexRay 2,5 MBit/s baud rate
	fcFRBaudRate2M5,
	//! FlexRay 5 MBit/s baud rate
	fcFRBaudRate5M,
	//! FlexRay 10 MBit/s baud rate
	fcFRBaudRate10M,
} fcFRBaudRate;

//! This structure describes the cluster configuration of a FlexRay communication 
//! controller.
//! \see #fcFRBaudRate, #fcChannel, #fcbFRSetCcConfiguration, #fcbFRGetCcConfiguration
typedef struct fcFRCcConfig
{
	//! The FlexRay baud rate. Valid values contains the enumeration #fcFRBaudRate.
	fcFRBaudRate BaudRate;
	//! Configures the action point offset in macroticks within static slots and
	//! symbol window. Valid values are 1 to 63 MT.
	fcDword gdActionPointOffset;
	//! Configures the upper limit of the acceptance window for a collision avoidance
	//! symbol (CAS). Valid values are 67 to 99 bit times.
	fcDword gdCASRxLowMax;
	//! Configures the duration of the dynamic slot idle phase. Has to be greater or
	//! equal than the idle detection time. Valid values are 0 to 2 Minislot.
	fcDword gdDynamicSlotIdlePhase;
	//! Configures the duration of a minislot in macroticks. Valid values are 2 to 63 MT.
	fcDword gdMinislot;
	//! Configures the action point offset in macroticks within the minislots of the
	//! dynamic segment. Valid values are 1 to 31 MT.
	fcDword gdMinislotActionPointOffset;
	//! Configures the duration of the Network Idle Time NIT at the end of the FlexRay
	//! communication cycle expressed in terms of microticks. Valid values are 2 to 805.
	fcDword gdNIT;
	//! Configures the duration of a static slot in macroticks. Valid values are 4
	//! to 659 MT.
	fcDword gdStaticSlot;
	//! Configures the duration of the Transmission Start Sequence (TSS) in terms
	//! of bit times (1 bit time = 4 uT = 100ns @ 10Mbps). Valid values are 3 to
	//! 15 bit times.
	fcDword gdTSSTransmitter;
	//! Configures the number of bit times used by the node to test the duration
	//! of the idle phase of the received wakeup symbol. Valid values are 14 to
	//! 59 bit times.
	fcDword gdWakeupSymbolRxIdle;
	//! Configures the number of bit times used by the node to test the duration
	//! of the low phase of the received wakeup symbol. Valid values are 10 to
	//! 55 bit times.
	fcDword gdWakeupSymbolRxLow;
	//! Configures the number of bit times used by the node to test the duration
	//! of the received wakeup pattern. Valid values are 76 to 301 bit times.
	fcDword gdWakeupSymbolRxWindow;
	//! Configures the number of bit times used by the node to transmit the idle
	//! phase of the wakeup symbol. Valid values are 45 to 180 bit times.
	fcDword gdWakeupSymbolTxIdle;
	//! Configures the number of bit times used by the node to transmit the low
	//! phase of the wakeup symbol. Valid values are 15 to 60 bit times.
	fcDword gdWakeupSymbolTxLow;
	//! Configures the maximum number of attempts that a cold starting node is
	//! permitted to try to start up the network without receiving any valid
	//! response from another node. Valid values are 2 to 31.
	fcDword gColdStartAttempts;
	//! Configures the upper limit for startup and wakeup listen timeout in
	//! the presence of noise expressed as a multiple of pdListenTimeout. Valid
	//! values are 2 to 16.
	fcDword gListenNoise;
	//! Configures the duration of one FlexRay communication cycle in macroticks.
	//! Valid values are 10 to 16000 MT.
	fcDword gMacroPerCycle;
	//! Defines the number of consecutive even / odd cycle pairs with missing
	//! clock correction terms that will cause a transition from NORMAL_ACTIVE
	//! or NORMAL_PASSIVE to HALT state. Valid values are 1 to 15 cycle pairs.
	fcDword gMaxWithoutClockCorrectionFatal;
	//! Defines the number of consecutive even / odd cycle pairs with missing
	//! clock correction terms that will cause a transition from NORMAL_ACTIVE
	//! to NORMAL_PASSIVE state. Valid values are 1 to 15 cycle pairs.
	fcDword gMaxWithoutClockCorrectionPassive;
	//! Configures the length of the network management vector. Valid values are
	//! 0 to 12 bytes.
	fcDword gNetworkManagementVectorLength;
	//! Configures the number of minislots within the dynamic segment of a FlexRay
	//! cycle. Valid values are 0 to 7986.
	fcDword gNumberOfMinislots;
	//! Configures the number of static slots in a FlexRay cycle. At least 2
	//! coldstart nodes must be configured to startup a FlexRay network. Valid
	//! values are 2 to 1023.
	fcDword gNumberOfStaticSlots;
	//! Determines the start of the offset correction within the NIT phase,
	//! calculated from start of cycle. Valid values are 8 to 15998 MT.
	fcDword gOffsetCorrectionStart;
	//! Configures the cluster-wide payload length for all frames sent in the
	//! static segment in double bytes. Valid values are 0 to 127.
	fcDword gPayloadLengthStatic;
	//! Configures the maximum number of frames within a cluster with sync frame
	//! indicator bit set to 1. Valid values are 2 to 15.
	fcDword gSyncNodeMax;
	//! Configures the number of microticks constituting the expanded range of
	//! measured deviation for startup frames during integration. Valid values
	//! are 0 to 1875 uT.
	fcDword pdAcceptedStartupRange;
	//! Configures wakeup / startup listen timeout in uT. Valid values are 1284
	//! to 1283846.
	fcDword pdListenTimeout;
	//! Configures the maximum drift offset between two nodes that operate with
	//! unsynchronized clocks over one FlexRay communication cycle in micorticks.
	//! Valid values are 2 to 1923 uT.
	fcDword pdMaxDrift;
	//! Controls the transition to HALT state due to a clock synchronization
	//! error. Valid values are 0 to 1. If a clock sync error occurred the CC
	//! will enter HALT state or enter/remain in NORMAL_PASSIVE state.
	fcDword pAllowHaltDueToClock;
	//! Defines the number of consecutive even / odd cycle pairs that must
	//! have valid clock correction terms before the CC is allowed to transit
	//! from NORMAL_PASSIVE to NORMAL_ACTIVE state. If set to 0 the CC is not
	//! allowed to transit from NORMAL_PASSIVE to NORMAL_ACTIVE state. Valid
	//! values are 0 to 31 even / odd cycle pairs.
	fcDword pAllowPassiveToActive;
	//! Selects the channel on which the CC sends a media test symbol. Valid
	//! values contains the enumeration #fcChannel.
	fcChannel pChannelsMTS;
	//! Selects the channel on which the CC is connected. Valid values are
	//! fcChannelA, fcChannelB and fcChannelBoth.
	fcChannel pChannels;
	//! Configures the cluster drift damping value used in clock synchronization
	//! to minimize accumulation of rounding errors. Valid values are 0 to 20 uT.
	fcDword pClusterDriftDamping;
	//! Configures the decoding correction value used to determine the primary
	//! time reference point. Valid values are 14 to 143 uT.
	fcDword pDecodingCorrection;
	//! Configuration to compensate the reception delays on channel A in
	//! microticks. Valid values are 0 to 200 uT.
	fcDword pDelayCompensationA;
	//! Configuration to compensate the reception delays on channel B in
	//! microticks. Valid values are 0 to 200 uT.
	fcDword pDelayCompensationB;
	//! Configures the external offset correction value in microticks to be applied
	//! by the internal clock synchronization algorithm. The value is subtracted or
	//! added to the calculated offset correction value. Valid values are 0 to 7 uT.
	fcDword pExternOffsetCorrection;
	//! Configures the external rate correction value in microticks to be applied
	//! by the internal clock synchronization algorithm. The value is subtracted or
	//! added to the calculated rate correction value. Valid values are 0 to 7 uT.
	fcDword pExternRateCorrection;
	//! Defines whether the key slot is used to transmit startup frames. NOT USED.
	fcDword pKeySlotUsedForStartup;
	//! Defines whether the key slot is used to transmit sync frames. NOT USED.
	fcDword pKeySlotUsedForSync;
	//! Configures the maximum minislot value allowed before inhibiting frame
	//! transmission in the dynamic segment of the cycle. Valid values are 0 to
	//! 7981 minislots. There is no transmission in dynamic segment if is set to 0. 
	fcDword pLatestTx;
	//! Configures the number of macroticks between the static slot boundary
	//! and the subsequent macrotick boundary of the secondary time reference
	//! point based on the nominal macrotick duration. Valid values are 2 to 72 MT.
	fcDword pMacroInitialOffsetA;
	//! Configures the number of macroticks between the static slot boundary
	//! and the subsequent macrotick boundary of the secondary time reference
	//! point based on the nominal macrotick duration. Valid values are 2 to 72 MT.
	fcDword pMacroInitialOffsetB;
	//! Configures the number of microticks between the actual time reference
	//! point on channel A and the subsequent macrotick boundary of the secondary
	//! time reference point. Valid values are 0 to 240 uT.
	fcDword pMicroInitialOffsetA;
	//! Configures the number of microticks between the actual time reference
	//! point on channel B and the subsequent macrotick boundary of the secondary
	//! time reference point. Valid values are 0 to 240 uT.
	fcDword pMicroInitialOffsetB;
	//! Configures the duration of the FlexRay communication cycle in microticks.
	//! Valid values are 640 to 640000 uT.
	fcDword pMicroPerCycle;
	//! Defines the maximum permitted offset correction value to be applied by
	//! the internal clock synchronization algorithm (absolute value). The CC
	//! checks only the internal offset correction value against the maximum
	//! offset correction value. Valid values are 5 to 15266 uT.
	fcDword pOffsetCorrectionOut;
	//! Defines the maximum permitted rate correction value to be applied by
	//! the internal clock synchronization algorithm. The CC checks only the
	//! internal rate correction value against the maximum rate correction value
	//! (absolute value). Valid values are 2 to 1923 uT.
	fcDword pRateCorrectionOut;
	//! Selects the initial transmission slot mode. Valid values are 0 to 1.
	//! In SINGLE slot mode the CC may only transmit the configured sync-frame
	//! message. In ALL slot mode the CC may transmit all configure Tx frames. 
	fcDword pSingleSlotEnabled;
	//! Selects the channel on which the CC sends the Wakeup pattern. Valid values
	//! are fcChannelA and fcChannelB.
	fcChannel pWakeupChannel;
	//! Configures the number of repetitions (sequences) of the Tx wakeup symbol.
	//! Valid values are 2 to 63.
	fcDword pWakeupPattern;
	//! Configures the external offset correction. Valid values are 0 to 3. Value 0
	//! and 1 means no external offset correction is used. Value 2 means the external
	//! offset correction value will be subtracted from calculated offset correction value.
	//! Value 3 means the external offset correction value will be added to calculated
	//! offset correction value.
	fcDword vExternOffsetControl;
	//! Configures the external rate correction. Valid values are 0 to 3. Value 0 and 1
	//! means no external rate correction is used. Value 2 means the external rate
	//! correction value will be subtracted from calculated rate correction value. Value
	//! 3 means the external rate correction value will be added to calculated rate
	//! correction value.
	fcDword vExternRateControl;
	//!< Reserved for future use
	fcDword Reserved[16];
} fcFRCcConfig;

#endif /* #define _FCBASE_TYPES_FLEXRAY_H_ */
