///////////////////////////////////////////////////////////////////////////////
//! \file fcBaseCAN.h:
//! \brief Header File for the fcBase CAN API.
//!
//! Use this header file for developing user defined FlexCard CAN applications.
//! See also printed documentation for more information. The data types and
//! enumerations can be found in the header file "fcBaseTypesCAN.h".
//!
//! Generally:
//!
//! - Each function of this library (except some error handling functions)
//!   returns an error code. If the return value is equal to 0, no error
//!   occured. A number greater than 0 indicates an error. To get more
//!   information about it, use the error handling functions.
//!
//! See also printed documentation for more information.
//!
//! \par Overview:
//! Author:         EBEL/WO/SH/Ba/TB/Tr/LJ
//! $Date: 2009-02-16 13:30:29 +0100 (Mo, 16 Feb 2009) $ <br>
//! \par Copyright:
//! Copyright 2008-2009 Eberspaecher Electronics GmbH & Co. KG
//! \version $Revision: 15011 $
///////////////////////////////////////////////////////////////////////////////

#ifndef _FCBASE_CAN_H_
#define _FCBASE_CAN_H_

#ifndef _FCBASE_TYPES_H_
	#error Please include first the header files "fcBaseTypes.h"
#endif 

#ifndef _FCBASE_TYPES_CAN_H_
	#error Please include first the header files "fcBaseTypesCAN.h"
#endif

// Linux
#ifdef linux

	#ifndef FCBASE_API
		#define FCBASE_API
	#endif

	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL extern "C" __attribute__ ((visibility ("default")))
	#else
		#define EXPORTDLL extern "C" 
	#endif

// Windows
#else

	#ifndef FCBASE_API
		#define FCBASE_API __stdcall
	#endif

	//! Please do not use the FCBASE_EXPORTS define in your application, unless
	//! you have the source code ;-)
	#ifdef FCBASE_EXPORTS
		#define EXPORTDLL extern "C" __declspec(dllexport)
	#else
		#define EXPORTDLL extern "C" __declspec(dllimport)
	#endif

#endif

//-----------------------------------------------------------------------------
//           Initialization 
//-----------------------------------------------------------------------------

//! This function is used to start the monitoring of a CAN bus. Once called,
//! the function changes the communication controller state from configuration
//! state to normal active state. The current communication controller state
//! can be read using the function #fcbGetCcState. If the FlexCard is started
//! the function #fcbGetCcState will return the value #fcStateNormalActive.
//! 
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the CAN communication controller.
//! \param restartTimestamps [IN] <br> Set this parameter to false to restart
//!		   the measurement without resetting the FlexCard timestamp. Set it to
//!		   true to start the measurement from the beginning. The timestamps
//!		   have micro second resolution.
//! \param mode [IN] <br> The monitoring mode. See #fcCANMonitoringMode for
//! 	   details which monitoring mode is supported.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \remarks After the monitoring has started, the user should check if the
//!          integration in the cluster was successful, #fcbCANGetCcState should
//!          return the state #fcCANStateNormalActive. 
//! \note After the monitoring has successfully started, the receive process has
//! 	  to be started as soon as possible to avoid an overflow (error packet
//! 	  fcErrFlexcardOverflow is received). Once an overflow occurred, no more
//! 	  packets can be received. The monitoring has to be stopped and started
//! 	  again.
//! \see #fcbCANMonitoringStop, #fcbCANGetCcState, #fcCANMonitoringMode
EXPORTDLL fcError FCBASE_API fcbCANMonitoringStart(fcHandle hFlexCard,
												   fcCC CC,
												   bool resetTimestamps,
												   fcCANMonitoringMode mode);


//! This function stops the CAN bus measurement of the selected communication
//! controller. The communication controller is set back in its configuration state.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the CAN communication controller.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to get
//!         extended error information.
//! \see #fcbCANMonitoringStart
EXPORTDLL fcError FCBASE_API fcbCANMonitoringStop(fcHandle hFlexCard,
												  fcCC CC);


//! This function returns the current CAN communication controller state. For a
//! description of possible states, refer to the enumeration #fcCANCcState.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> Index of the CAN communication controller.
//! \param pState [OUT] <br> Current CAN communication controller state.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         please use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbMonitoringStart, #fcbMonitoringStop
EXPORTDLL fcError FCBASE_API fcbCANGetCcState(fcHandle hFlexCard,
											  fcCC CC,
											  fcCANCcState* pState);


//-----------------------------------------------------------------------------
//          Configuration
//-----------------------------------------------------------------------------

//! This function configures a CAN communication controller. This function
//! cannot be called during monitoring. Before the configuration of the CAN
//! CC starts, all CAN message buffers are reset.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> CAN communication controller identifier.
//! \param cfg [IN] <br> CAN communication controller configuration parameters.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcCANCcConfig
EXPORTDLL fcError FCBASE_API fcbCANSetCcConfiguration(fcHandle hFlexCard,
													  fcCC CC,
													  fcCANCcConfig cfg);


//! This function configures the message buffers of the CAN communication controller.
//! Configuring message buffers is allowed in all communication controller states.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> CAN communication controller identifier.
//! \param bufferNumber [IN] <br> Identifier of the message buffer to be configured.
//! \param cfg [IN] <br> Message buffer configuration parameters.
//! \param ignoreTxReqLock [IN] <br> Set to true to configure the message buffer also
//!                                  if the previous transmittable buffer was not 
//!                                  transmitted yet (TxRqst still = 1)
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbCANGetMessageBuffer, #fcCANBufCfg
EXPORTDLL fcError FCBASE_API fcbCANSetMessageBuffer(fcHandle hFlexCard,
													fcCC CC,
													fcDword bufferNumber,
													fcCANBufCfg cfg,
													bool ignoreTxRqstLock);


//! This function reads a specific CAN message buffer configuration.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> CAN communication controller identifier.
//! \param bufferNumber [IN] <br> Identifier of the message buffer to be read.
//! \param cfg [OUT] <br> The configuration parameters of the specified message buffer.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbCANSetMessageBuffer, #fcCANBufCfg
EXPORTDLL fcError FCBASE_API fcbCANGetMessageBuffer(fcHandle hFlexCard,
													fcCC CC,
													fcDword bufferNumber,
													fcCANBufCfg* pCfg);


//-----------------------------------------------------------------------------
//          Transmit
//-----------------------------------------------------------------------------

//! This function writes the data bytes in a CAN communication controller transmit
//! buffer of the FlexCard. The message should normally be transmitted as soon as
//! possible. In case the transmission of any number of message buffers may be
//! requested at the same time, they are transmitted subsequently according to their
//! priority (The message object numbers are configurable from 1 up to 128, the lower
//! is the message number, the higher is the priority). If the transmit acknowledgment
//! is activated, a CAN packet with a direction flag = 1 (Tx) is generated as soon as
//! the message has been transmitted. This function should only be called when the
//! FlexCard is in normal active state or when all message buffer configurations have
//! been done.
//! \param hFlexCard [IN] <br> Handle to a FlexCard.
//! \param CC [IN] <br> CAN communication controller identifier.
//! \param bufferNumber [IN] <br> The number of the message buffer used for the transmission.
//! \param data [IN] <br> The data to be transmitted. The configured DLC in the message
//!						  buffer determinates the size of bytes which will be copied in
//!						  the transmit buffer.
//! \param transmitNewData [IN] <br> Set to true to update the data of the message
//!		   buffer. Set to false if the previous data shall be sent again.
//! \param ignoreTxReqLock [IN] <br> Set to true to transmit the message although
//!                                  if the previous message was not transmitted yet.
//! \return If the function succeeds, the return value is 0. If the value <> 0,
//!         use the functions described in the section Error Handling to
//!         get extended error information.
//! \see #fcbCANGetMessageBuffer, #fcCANBufCfg
EXPORTDLL fcError FCBASE_API fcbCANTransmit(fcHandle hFlexCard,
											fcCC CC,
											fcDword bufferNumber,
											fcByte data[8],
											bool transmitNewData,
											bool ignoreTxRqstLock);

#endif /* #define _FCBASE_CAN_H_ */
